#include <layout.glsl>
#include "raytracing.glsl"
#include "primitives.glsl"
#include "smoke.glsl"
#include "zebra.glsl"

vec3 skyboxColor(Ray ray) {
    vec3 skyGradient = mix(sky.bottomColor.xyz, sky.topColor.xyz, .5 + .5 * ray.direction.y);
    vec3 sky = skyGradient * smokeParams.skyBlend;
    vec3 smoke = smoke(ray.direction.xy) * smokeParams.smokeBlend;
    vec3 zebra = zebra(ray.direction.xy) * smokeParams.zebraBlend;
    return sky + smoke + zebra;
}

float skyboxEmission(Ray ray) {
    return mix(sky.bottomColor.w, sky.topColor.w, .5 + .5 * ray.direction.y);
}

float centeredIndex(int count, int index) {
  return float(index) - 0.5 * (float(count) - 1.0);
}

float dampedOsc(float t, float k, float w) { return exp(-k * max(t, 0.0)) * cos(w * t); }

float shutterTime;


// MorphVM Primitives
const int MorphPrimitive = 0;
const int MorphPrimitiveBall = 0;
const int MorphPrimitiveIcosahedron = 1;
const int MorphPrimitiveBox = 2;
const int MorphPrimitiveLast = 2;

// MorphVM Operators;
const int MorphOp = 0;
const int MorphOpMin = 0;
const int MorphOpMax = 1;
const int MorphOpSmoothMin = 2;
const int MorphOpLerp = 3;
const int MorphOpTorsion = 4;

vec4 transform(vec3 point, float t, mat4 matrixNow, mat4 matrixPrev) {
    // Translation
    vec3 translation = mix(matrixNow[3].xyz, matrixPrev[3].xyz, t);
    matrixNow[3][0] = translation.x;
    matrixNow[3][1] = translation.y;
    matrixNow[3][2] = translation.z;
    float scale = length(matrixNow[0].xyz);

    // Transformation
    point = (matrixNow * vec4(point, 1.0)).xyz;

    // Result
    return vec4(point, scale);
}

vec3 torsion(vec3 point, float amount) {
    return (rotate4d(vec3(0, 1, 0), point.y * amount) * vec4(point, 1.0)).xyz;
}

vec4 sdfScene(vec3 point) {
    point = torsion(point, morphers[0].fParams.w).xyz;

    int ic = morphersParameters.count;
    vec4 result = vec4(MAXDISTANCE, 0.0, 0.0, 0.0);
    result.y = morphers[0].materialId + 1.0;
    result.z = result.y;
    float t = (time.current - shutterTime) / shutter.deltaTime;
    vec4 transformedPoint;
    float radius;

    switch (morphers[0].morphCode) {
        case MorphPrimitiveBall:
            transformedPoint = transform(point, t, morphers[0].matrix, morphers[0].previousMatrix);

            radius = morphers[0].fParams.x;
            float bigBall = transformedPoint.w * ball(transformedPoint.xyz, radius, vec3(0.0));
            result.x = bigBall;
            float secondaryTag = result.x;

            for (int index = 1; index < ic; index++) {
                // Transform
                vec4 transformedPoint = transform(point, t, morphers[index].matrix, morphers[index].previousMatrix);
                float radius = morphers[index].fParams.x;
                // Render
                float ball = transformedPoint.w * ball(transformedPoint.xyz, radius, vec3(0.0));
                float hedron = transformedPoint.w * icosahedron(transformedPoint.xyz, radius);
                float morpher = mix(ball, hedron, morphers[index].fParams.y);
                float blend = smoothMin(result.x, morpher, morphers[index].fParams.z);
                result.x = blend;

                if (morpher < secondaryTag)
                    secondaryTag = morpher;
                    result.z = morphers[index].materialId + 1.0;
            }

            float denom = bigBall + secondaryTag;
            result.w = 1.0 - (denom < EPSILON ? 0.5 : secondaryTag / denom);

            float blend = morphers[0].fParams.z;

            if (blend > 0.0) {
                vec4 transformedPoint = transform(point, t, morphers[0].matrix, morphers[0].previousMatrix);
                float radius = morphers[0].fParams.x;
                float hedron = transformedPoint.w * icosahedron(transformedPoint.xyz, radius);
                result.x = mix(result.x, hedron, blend);
            }

            break;
        case MorphPrimitiveIcosahedron:
            for (int index = 1; index < ic; index++) {
                vec4 transformedPoint;
                switch (morphers[index].morphCode) {
                    case MorphPrimitiveIcosahedron:
                        // Transform hedron
                        transformedPoint = transform(point, t, morphers[index].matrix, morphers[index].previousMatrix);
                        // Render hedron
                        float radius = morphers[index].fParams.x;
                        float hedron = transformedPoint.w * icosahedron(transformedPoint.xyz, radius);

                        // Transform slice
                        transformedPoint = transform(point, t, morphers[index+1].matrix, morphers[index+1].previousMatrix);
                        // Render slice
                        float slicer = transformedPoint.w * box(transformedPoint.xyz, morphers[index+1].fParams.xyz);
                        float slice = max(hedron, slicer);
                        result.x = min(slice, result.x);
                        index++;
                        break;
                    case MorphPrimitiveBox:
                        // Transform box
                        transformedPoint = transform(point, t, morphers[index].matrix, morphers[index].previousMatrix);
                        // Render box
                        float box = transformedPoint.w * box(transformedPoint.xyz, morphers[index].fParams.xyz);
                        result.y = box < result.x ? morphers[index].materialId + 1.0 : result.y;
                        result.x = min(box, result.x);
                        break;
                }
            }

            break;
    }

    return result;
}
